<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Registration;
use App\Category;
use App\Exports\RegistrationsExport;
use Maatwebsite\Excel\Facades\Excel;
use Carbon\Carbon;

class RegistrationController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function __construct()
    {
        $this->middleware('auth');
    }
    
    public function index(Request $request)
    {
        $sideBar = 'Registrations';

        $result = Registration::orderBy('id', 'DESC');

        if($request->search != ''){
            $result->where(function ($query) use ($request) {
                $query->where('name', 'LIKE', '%'.$request->search.'%')
                    ->orWhere('email', 'LIKE', '%'.$request->search.'%')
                    ->orWhere('phone', 'LIKE', '%'.$request->search.'%');
            });
        }

        $lists = $result->paginate(20);

        if ($request->ajax()) {
            return view('admin.registrations.ajax-list', ['lists' => $lists])->render();    
        }

        $categories = Category::where('status', 1)->orderBy('name', 'ASC')->get();

        return view('admin.registrations.list', compact('sideBar', "lists", "categories"));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return back();
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        return back();
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        return back();
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $sideBar = 'Registrations';

        $data = Registration::find($id);

        $categories = Category::where('status', 1)->orderBy('name', 'ASC')->get();

        return view('admin.registrations.edit', compact('sideBar', 'data', 'categories'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $this->validate(request(), [
            'name' => 'required',
            'email' => 'required',
            'phone' => 'required',
            'category' => 'required',
            'age_group' => 'required',
        ]);

        $name = $request->name;
        $email = $request->email;
        $phone = $request->phone;
        $category = $request->category;
        $age_group = $request->age_group;

        $update = Registration::find($id);
        $update->name = $name;
        $update->email = $email;
        $update->phone = $phone;
        $update->category = $category;
        $update->age_group = $age_group;
        $update->updated_at = Carbon::now();
        $update->save();

        return back();
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $destroy = Registration::findOrFail($id);
        $destroy->delete();
        
        return back();
    }

    public function export(Request $request) 
    {
        $category = $request->category;
        $from = $request->from_date;
        $to = $request->to_date;

        return Excel::download(new RegistrationsExport($category, $from, $to), 'Registrations.csv');
    }

    public function exportCount(Request $request)
    {
        $category = $request->category;
        $from = $request->from_date;
        $to = $request->to_date;

        $result = Registration::orderBy('updated_at', 'DESC');

        if($category != '' && $category != 'all'){
            $result->where(function ($query) use ($category){
                $query->where('category', $category);
            });
        }

        if($from != '' && $to != ''){
            if(strtotime($from) == strtotime($to)){
                $result->where(function ($query) use ($from, $to){
                    $query->whereDate('updated_at', $from);
                });
            }
            else{
                $result->where(function ($query) use ($from, $to){
                    $query->whereBetween('updated_at', [$from, $to]);
                });
            }
        }

        $registration = $result->count();

        return $registration;
    }
}
